//--------------------------------------
//       StdioLib shared library 
//            Version 2.03
//          by Harry Konstas 
//        & Philippe Guillot 
//           Mar. 25, 2002 
//-------------------------------------- 
 
// History
// 25/03/02 STDOUT included in lib

// !! change code record 0 into "libr 0" with RsrcEdit !! 

#pragma segment 0 
 
// POSIX error codes 
#define EIO       15  // i/o error 
#define ENOENT    22  // no such file or directory 
#define ENOMEM    25  // not enought memory 
#define ENOSPC    26  // no space on drive 
#define ENOSYS    27  // not imlemented 

#define EOF  -1

#define DocBufSize 2048
#define UncBufSize 6144

#define COMPRESSED 4
#define READ_MODE  1
#define WRITE_MODE 2
#define CONSOLE    8
 
// console data type 
typedef struct 
{ 
  int x; 
  int y; 
  int w; 
  int h; 
  int hgt; 
  int wdt; 
  int colstart; 
  int colend; 
  int rowstart; 
  int rowend; 
  int curcol; 
  int currow;
  char history[80]; 
}STDOUT; 
 
typedef struct { 
  int compressed; 
  int spare; 
  ULong DocSize; 
  UInt rNum; 
  UInt rSize; 
  ULong position; 
  int spare2;         // SmartDoc 
}DOCHEADER; 
 
typedef struct 
{ 
  UInt16 mode;
/*
 b0 read mode
 b1 write mode
 b2 compressed
 b3 console
*/
  DOCHEADER *pHeader; // pointer to header ( new doc mode ) 
  DmOpenRef db;      // database reference 
  int iRec;          // current record ( read mode ) 
  int nRecs;         // read mode : record number 
                     // new doc   : -1 
                     // console   : -2 
  ULong  RecSize;    // char left to read in current record ( read mode ) 
  Handle hFile;      // handle to current record ( read mode ) 
  Byte   *pData;     // pointer to data ( read mode ) 
  VoidHand hBuff;    // handle to buffer ( write mode ) 
  int iBuff;         // char number in buffer ( new doc  mode ) 
  char *buffer;      // pointer to buffer ( new doc mode ) 
}DOCFILE; 
 
typedef struct 
{ 
 Ptr* dispatchTblP; 
 void* globalsP; 
 LocalID dbID; 
 VoidPtr CodeRscH; 
}SysLibTblEntryType; 
 
typedef SysLibTblEntryType* SysLibTblEntryPtr; 
SysLibTblEntryPtr SysLibTblEntry(UInt) SYS_TRAP(sysTrapSysLibTblEntry); 
 
typedef struct 
{ 
 UInt      refcount;
 Handle    hTinyFont;
 Handle    hStdFont;
 DmOpenRef db;
 STDOUT    Out;
} stdio_globals; 
 
void*getTable(); 

// Various API Prototypes 
void RctSetRectangle(RectanglePtr,SWord,SWord,SWord,SWord) SYS_TRAP(sysTrapRctSetRectangle); 
void InsPtSetLocation(int,int) SYS_TRAP(sysTrapInsPtSetLocation); 
void InsPtSetHeight(int) SYS_TRAP(sysTrapInsPtSetHeight); 
void InsPtEnable(Boolean) SYS_TRAP(sysTrapInsPtEnable); 
Err MemPtrSetOwner(MemPtr,UInt16) SYS_TRAP(sysTrapMemPtrSetOwner); 
 
 
//-------------------------------------- 
//      library dispatch functions 
//-------------------------------------- 
 
Err Start(UInt refnum, SysLibTblEntryPtr entry_p) 
{ 
  entry_p->dispatchTblP=getTable(); 
  entry_p->globalsP=NULL; 
 
  return 0; 
} 
 
void table(); 
 
asm void*getTable() 
{ 
  lea table_00(pc),a0 
  rts 
} 

asm void table() 
{ 
  dc.w 322 ; 8*n + 2 
  dc.w 82  ; 2*n + 2 
  dc.w 88  ; + 6 
  dc.w 94  ; + 6 
  dc.w 100 
  dc.w 106 
  dc.w 112 
  dc.w 118 
  dc.w 124 
  dc.w 130 
  dc.w 136 
  dc.w 142 
  dc.w 148 
  dc.w 154 
  dc.w 160 
  dc.w 166 
  dc.w 172 
  dc.w 178 
  dc.w 184 
  dc.w 190
  dc.w 196
  dc.w 202
  dc.w 208
  dc.w 214
  dc.w 220
  dc.w 226
  dc.w 232
  dc.w 238
  dc.w 244
  dc.w 250
  dc.w 256
  dc.w 262
  dc.w 268
  dc.w 274
  dc.w 280
  dc.w 286
  dc.w 292
  dc.w 298
  dc.w 304
  dc.w 310
  dc.w 316

  ; 'jmp' emulation 
  pea stdioopen_00(pc) 
  rts 
  pea stdioclose_00(pc) 
  rts 
  pea stdiosleep_00(pc) 
  rts 
  pea stdiowake_00(pc) 
  rts 
  pea initstdout_00(pc) 
  rts 
  pea clearstdout_00(pc) 
  rts 
  pea printstdout_00(pc) 
  rts 
  pea getstdin_00(pc) 
  rts 
  pea closestdout_00(pc) 
  rts 
  pea putstdout_00(pc) 
  rts 
  pea getstdkey_00(pc) 
  rts 
  pea revstr_00(pc) 
  rts 
  pea longtoa_00(pc) 
  rts 
  pea newdoc_00(pc) 
  rts 
  pea opendoc_00(pc) 
  rts 
  pea closedoc_00(pc) 
  rts 
  pea writedoc_00(pc) 
  rts 
  pea readdocline_00(pc) 
  rts
  pea getfonts_00(pc)
  rts
  pea openstdin_00(pc)
  rts
  pea closestdin_00(pc)
  rts
  pea getcharin_00(pc)
  rts
  pea deletedb_00(pc)
  rts
  pea newdb_00(pc)
  rts
  pea opendb_00(pc)
  rts
  pea fwritedoc_00(pc)
  rts
  pea freadoc_00(pc)
  rts
  pea GotoXY_00(pc)
  rts
  pea WriteDocChar_00(pc)
  rts
  pea openstdout_00(pc)
  rts
  pea FixToStr_00(pc)
  rts
  pea FPutS_00(pc)
  rts
  pea FGetS_00(pc)
  rts
  pea StrToL_00(pc)
  rts
  pea StrToD_00(pc)
  rts
  pea dummy_00
  rts
  pea dummy_00
  rts
  pea dummy_00
  rts
  pea dummy_00
  rts
  pea dummy_00
  rts
  dc.b "StdioLib" 
} 
 
//-------------------------------------- 
//         Library open / close 
//-------------------------------------- 


Err dummy(UInt refnum)
{ // dummy function for unused lib slots
 return 0;
}

Err stdioopen(UInt refnum) 
{ 
 
  SysLibTblEntryPtr ep; 
  stdio_globals*gl;
  LocalID id;
 
  ep=SysLibTblEntry(refnum); 
  gl=(stdio_globals*)ep->globalsP; 
 
  if(gl) 
  { 
    gl->refcount++; 
    return 0; 
  } 
  ep->globalsP=MemPtrNew(sizeof(stdio_globals)); 
  if (ep->globalsP==NULL) return 1; 
 
  MemPtrSetOwner(ep->globalsP,0); 
  gl=(stdio_globals*)ep->globalsP; 
  gl->refcount=1;

  id=DmFindDatabase(0,"StdioLib");
  if(id) gl->db=DmOpenDatabase(0,id,dmModeReadOnly);
  else return 1;
  if(gl->db==NULL) return 1;
  gl->hTinyFont=DmGetResource('tFNT',1000);
  if(gl->hTinyFont==NULL) return 1;
  gl->hStdFont=DmGetResource('tFNT',1001);
  if(gl->hStdFont==NULL) return 1;

  return 0;
 
} 
 
Err stdioclose(UInt refnum,UIntPtr numappsP) 
{ 
 
  SysLibTblEntryPtr ep; 
  stdio_globals* gl; 
 
  ep=SysLibTblEntry(refnum); 
  gl=(stdio_globals*)ep->globalsP; 
 
  if(!gl) return 1; 
  *numappsP=--gl->refcount; 
 
  if(*numappsP==0) 
  { 
    DmReleaseResource(gl->hTinyFont);
    DmReleaseResource(gl->hStdFont); 
    DmCloseDatabase(gl->db);

    MemPtrFree(ep->globalsP); 
    ep->globalsP=NULL;
  } 
 
  return 0; 
 
} 
 
Err stdiosleep(UInt refnum) 
{ 
  return 0; 
} 
 
Err stdiowake(UInt refnum) 
{ 
  return 0; 
} 
 
//-------------------------------------- 
//          Various utilities 
//-------------------------------------- 
 
static void ScrollUp(STDOUT *pOut) 
{ 
 
  RectangleType aRec, bRec; 
  
  RctSetRectangle(&aRec,pOut->x,pOut->y,pOut->w,pOut->h); 
  pOut->currow=pOut->rowend; 
  WinScrollRectangle(&aRec,up,pOut->hgt,&bRec); 
  WinEraseRectangle(&bRec,0); 
 
} 
 
static void SetCursorPos(STDOUT *pO) 
{ 
  InsPtSetLocation(pO->curcol*pO->wdt-1,pO->currow*pO->hgt); 
}

#define controlKeyMask 0x20

static int TranslateKey(EventPtr pe)
{
 int k;
 k=pe->data.keyDown.chr;
 // handle control key
 if(pe->data.keyDown.modifiers&controlKeyMask)
 {
  if((k>='a')&&(k<='z')) k-='a'-1;
  if((k>='A')&&(k<='Z')) k-='A'-1;
 }
 return k;
}

 
Err revstr(UInt refnum,char *s) 
{ 
 
 int i,j; 
 char c; 
 i=0; 
 j=StrLen(s)-1; 
 while(i<j) 
 { 
  c=s[i]; s[i]=s[j]; s[j]=c; 
  i++; j--; 
 } 
 return 0; 
} 
 
Err longtoa(UInt refnum,long value, char *string,UInt16 radix) 
{ 
 UInt32 t; 
 int neg; 
 UInt16 dig; 
 char*ptr; 
 if (value<0) { neg=1; t=-value; } 
 else { neg=0; t=value; } 
 ptr = string; 
 
 do  
 { 
  dig=t%radix; 
  *(ptr++)=(char)dig+(dig<10?'0':'a'-10); 
 } while ((t/=radix) != 0); 
 if(neg) *ptr++ = '-'; 
 *ptr = 0; 
 return revstr(refnum,string); 
 
} 
 
Err getfonts(UInt16 refnum,Handle*hTinyFont,Handle*hStdFont)
{
  SysLibTblEntryPtr ep; 
  stdio_globals* gl; 
 
  ep=SysLibTblEntry(refnum); 
  gl=(stdio_globals*)ep->globalsP; 
 
  *hTinyFont=gl->hTinyFont;
  *hStdFont=gl->hStdFont;

  return 0;
}

int decompress(Byte*d,Byte *o,int l)
{ 
 int i,j; 
 int c; 

 i=0;
 j=0;
 while (i < l)
 { 
  c=o[i++]; 
	 
  if ((c > 0) && (c < 9))
  {	// n chars copyed from in to out
    while (c-- > 0) d[j++] = o[i++]; 
  } 
  else 
  { 
   if (c < 0x80) 
   { // literal 
    d[j++]=c; 
   } 
   else 
   { 
    if (c >= 0xc0)
    { // space preceded literal, high bit off
     d[j++]=' ';
     d[j++]=c & 0x7F; 
    } 
    else
    { // duplication - <0b10><11 bit back displacement><3 bit length (plus 3)> 
     int orig; 
     int count; 
     c=(c <<= 8) + o[i++]; 
     orig = j - ((c & 0x3fff) >> 3); 
     count = (c & 0x7) + 3;
     while (count--) d[j++] = d[orig++]; 
    } 
   } 
  }
 } 
 return j; 
} 
 
// return integral value of digit c
int radix(int c)
{
  if(c>='0'&&c<='9') return c-'0';
  if(c>='a'&&c<='z') return c-'a'+10;
  if(c>='A'&&c<='Z') return c-'A'+10;
  return 36;
}

// 10^n
double Pow10(UInt16 n) 
{
 UInt16 mask;
 double y;
 
 if(!n) return 1.0;
 mask=0x8000;
 y=10.0;
 while(mask>n) mask>>=1;
 for(mask>>=1;mask;mask>>=1)
 {
  y=y*y;
  if(n&mask) y=y*10.0;
 }
 return y;
}

//-------------------------------------- 
//         STDIN/OUT functions 
//-------------------------------------- 
 
Err initstdout(UInt refnum,FontID*fID,int x,int y,int w,int h) 
{ 
 
 STDOUT *pO; 
 RectangleType Rec; 
 SysLibTblEntryPtr ep; 
 stdio_globals* gl; 
 
 ep=SysLibTblEntry(refnum); 
 gl=(stdio_globals*)ep->globalsP; 
 
 *fID=FntSetFont(*fID);
 pO=&gl->Out;
 pO->x=x; 
 pO->y=y; 
 pO->w=w; 
 pO->h=h; 
 pO->history[0]=0; 
 pO->hgt=FntCharHeight(); 
 pO->wdt=FntCharWidth('M'); 
 
 pO->curcol=(pO->colstart=x/pO->wdt); 
 pO->currow=(pO->rowstart=y/pO->hgt); 
 
 pO->colend=pO->colstart+(w/pO->wdt)-1; 
 pO->rowend=pO->rowstart+(h/pO->hgt)-1; 
 
 RctSetRectangle(&Rec,x,y,w,h); 
 WinEraseRectangle(&Rec,0); 
 WinSetClip(&Rec); 
 
 return 0; 
 
} 
 
Err clearstdout(UInt refnum) 
{ 
 
  RectangleType Rec; 
  SysLibTblEntryPtr ep; 
  stdio_globals* gl; 
  STDOUT*pO;
 
  ep=SysLibTblEntry(refnum); 
  gl=(stdio_globals*)ep->globalsP; 
  pO=&gl->Out;

  RctSetRectangle(&Rec,pO->x,pO->y,pO->w,pO->h); 
  WinEraseRectangle(&Rec,0); 
  pO->curcol=pO->colstart; 
  pO->currow=pO->rowstart; 
  return 0; 
 
} 

Err GotoXY(UInt16 refnum,int x,int y)
{
  SysLibTblEntryPtr ep; 
  stdio_globals* gl; 
 
  ep=SysLibTblEntry(refnum); 
  gl=(stdio_globals*)ep->globalsP; 
  gl->Out.x=x;
  gl->Out.y=y;
  return 0;
}

Err printstdout(UInt refnum,char*s) 
{ 
 
  int i,x,y; 
  SysLibTblEntryPtr ep; 
  stdio_globals* gl; 
 
  ep=SysLibTblEntry(refnum); 
  gl=(stdio_globals*)ep->globalsP; 
  STDOUT*pO;

  pO=&gl->Out; 
  i=0; 
  
  while(s[i]) 
  { 
    if(s[i]==13) // carriage return 
    { 
      pO->curcol=pO->colstart; 
      i++; 
      continue; 
    } 
 
    if(s[i]==10) // line feed + carriage return 
    { 
       pO->curcol=pO->colstart; 
       pO->currow++; 
       if(pO->currow>pO->rowend) 
       ScrollUp(pO); 
       i++; 
       continue; 
    } 
 
    if(pO->curcol>pO->colend) 
    { 
       pO->curcol=pO->colstart; 
       pO->currow++; 
    } 
 
    if(pO->currow>pO->rowend) 
      ScrollUp(pO); 
 
    WinDrawChars(s+i,1,pO->curcol*pO->wdt,pO->currow*pO->hgt); 
    pO->curcol++; 
    i++; 
 
  } 
 
  return 0; 
 
} 
 
Err putstdout(UInt refnum,char c) 
{ 
 
  char s[2]; 

  *s=c; 
  s[1]=0; 
 
  return printstdout(refnum,s); 
 
} 
 
Err getstdin(UInt refnum,char*s) 
{ 
 
  EventType evt; 
  int k,kcount; 
  SysLibTblEntryPtr ep; 
  stdio_globals* gl; 
  STDOUT *pOut;
 
  ep=SysLibTblEntry(refnum); 
  gl=(stdio_globals*)ep->globalsP; 
  pOut=&gl->Out;
 
  kcount=0; 
  s[0]=0; 
 
  SetCursorPos(pOut); 
  InsPtSetHeight(pOut->hgt); 
  InsPtEnable(1); 
 
  while(1) 
  { 
    EvtGetEvent(&evt,evtWaitForever); 
    SysHandleEvent(&evt); 
 
    if(evt.eType==keyDownEvent) 
    { 
      k=TranslateKey(&evt); 
      if(k==7) 
      { 
        SndPlaySystemSound(1); 
        continue; 
      } 
 
      // hard buttons 
      if(k==264||(k>515&&k<520)) 
      { 
        s[0]=-1; s[1]=0; 
        return 0; 
      } 
 
      // Return 
      if(k==10) 
      { 
        s[kcount]=0; 
        InsPtEnable(0); 
        putstdout(refnum,'\n'); 
        break; 
      } 
 
      // up button or up arrow 
      if(k==11||k==30) 
      { 
        StrCopy(s,pOut->history); 
        kcount=StrLen(s); 
        printstdout(refnum,pOut->history); 
        SetCursorPos(pOut);
        continue;
      } 
 
      // backspace 
      if(k==8) 
      { 
        if(kcount) 
        { 
          if(pOut->curcol>0) 
          { 
            pOut->curcol--; 
          } 
          else 
          { 
            pOut->currow--; 
            pOut->curcol=pOut->colend; 
          } 
 
          putstdout(refnum,' '); 
          pOut->curcol--; 
          s[--kcount]=0; 
          SetCursorPos(pOut); 
        }
        continue;
      } 
 
      if(kcount>=79) 
      { 
        SndPlaySystemSound(1); 
        continue; 
      } 
      if(k>255) continue;
      s[kcount++]=k; 
      InsPtEnable(0); 
      putstdout(refnum,k); 
      SetCursorPos(pOut); 
      InsPtEnable(1);
 
    } // key down event 
 
  } 
 
  StrCopy(pOut->history,s); 
  return 0; 
} 
 
Err openstdout(UInt refnum,DOCFILE**ppf)
{
 DOCFILE*pf;
 pf=MemPtrNew(sizeof(DOCFILE));
 if(pf==NULL) return ENOMEM;
 pf->nRecs=-2;
 *ppf=pf;
 return 0;
}


Err closestdout(UInt refnum,DOCFILE*pf)
{
 MemPtrFree(pf);
 return 0;
} 
 
Err getstdkey(UInt refnum,int *k) 
{ 
 
  EventType evt; 
 
  while(1) 
  { 
     EvtGetEvent(&evt,evtWaitForever); 
     SysHandleEvent(&evt); 
     if(evt.eType==keyDownEvent) 
     { 
       *k=TranslateKey(&evt); 
       return 0;
     } 
  } 
} 
 
// ------------------------------------- 
//     File stream (DOC) functions 
// ------------------------------------- 

Err GetDocChar(DOCFILE*pDoc,int*c)
{
  Byte*p; // temporary pointer to record
  if(pDoc->RecSize==0)
  {
    if(++pDoc->iRec >= pDoc->nRecs)
    {
      *c=EOF;
      return 0;
    }
    if(pDoc->hFile) MemHandleUnlock(pDoc->hFile);
    pDoc->hFile = DmQueryRecord(pDoc->db,pDoc->iRec);
    if(pDoc->hFile==NULL)
    {
      *c=EOF;
      return EIO;
    }
    if (pDoc->mode&COMPRESSED)
    {
     p=MemHandleLock(pDoc->hFile);
     pDoc->RecSize=decompress(pDoc->buffer,p,MemHandleSize(pDoc->hFile));
    }
    else
    {
     pDoc->RecSize = MemHandleSize(pDoc->hFile);
     pDoc->buffer = MemHandleLock(pDoc->hFile);
    }
    pDoc->iBuff=0;
  }
    
  *c = (unsigned int)pDoc->buffer[pDoc->iBuff++];
  pDoc->RecSize--;

  return 0; 
}

Err WriteDocChar(UInt16 refnum,DOCFILE *pDoc, unsigned char c) 
{ 
 
  VoidPtr p; 
  int n; 
 
  pDoc->buffer[pDoc->iBuff++] = c; 
 
  // flush + new record 
  if(pDoc->iBuff>=DocBufSize)  
  { 
    n=-1; 
       
    pDoc->hFile = DmNewRecord(pDoc->db,&n,StrLen(pDoc->buffer)); 
    if(pDoc->hFile==NULL) return ENOSPC; 
  
    p = MemHandleLock(pDoc->hFile); 
    DmWrite(p,0,pDoc->buffer,pDoc->iBuff); 
    MemHandleUnlock(pDoc->hFile); 
    DmReleaseRecord(pDoc->db,n,true); 
 
    pDoc->pHeader->DocSize += pDoc->iBuff; 
    pDoc->iBuff=0; 
    *pDoc->buffer=0; 
 
  } 

  return 0; 
} 
 
Err closedoc(UInt16 refnum, DOCFILE *pDoc) 
{ 
  VoidPtr p; 
  int n=0; 
 
  // check console mode 
  if(pDoc->nRecs==-2) return 0; 
 
  // check if new Doc mode  
  if(pDoc->nRecs==-1) 
  { 
 
    pDoc->pHeader->rNum = DmNumRecords(pDoc->db); 
 
    // flush buffer 
    if(pDoc->iBuff>0) 
    { 
 
      n = -1; // new record 
      //pDoc->buffer[pDoc->iBuff] = 0; 
      pDoc->pHeader->DocSize += pDoc->iBuff; 
      pDoc->hFile = DmNewRecord(pDoc->db,&n,pDoc->iBuff); 
 
      if (pDoc->hFile==NULL) 
      { 
        return ENOSPC; 
      } 
 
      p = MemHandleLock(pDoc->hFile); 
      DmWrite(p,0,pDoc->buffer,pDoc->iBuff); 
      MemHandleUnlock(pDoc->hFile); 
      DmReleaseRecord(pDoc->db,n,true); 
      pDoc->pHeader->rNum++; 
 
    } 
 
    // write header 
    n = 0;  
    pDoc->hFile = DmNewRecord(pDoc->db,&n,sizeof(DOCHEADER)); 
 
    if (pDoc->hFile==NULL) 
    { 
      return ENOSPC; 
    } 
 
    p = MemHandleLock(pDoc->hFile); 
    DmWrite(p,0,pDoc->pHeader,sizeof(DOCHEADER)); 
    MemHandleUnlock(pDoc->hFile); 
    DmReleaseRecord(pDoc->db,n,true); 
    MemPtrFree(pDoc->pHeader); 
 
    MemHandleUnlock(pDoc->hBuff); 
    MemHandleFree(pDoc->hBuff); 
  } 
  else
  {
    MemHandleUnlock(pDoc->hFile); 
    if(pDoc->mode&COMPRESSED) MemPtrFree(pDoc->buffer);
  }
  DmCloseDatabase(pDoc->db); 
  MemPtrFree(pDoc); 
 
  return 0; 
} 

// open doc in read mode
Err opendoc(UInt16 refnum,DOCFILE**pO,CharPtr fname) 
{ 
 
  LocalID id; 
  DOCFILE *pDoc; 
 
  pDoc = MemPtrNew(sizeof(DOCFILE)); 
  *pO=pDoc; 
  if (pDoc==NULL) return ENOMEM; 
 
  pDoc->nRecs = 0; 

  id=DmFindDatabase(0,fname); 
  pDoc->db = id?DmOpenDatabase(0,id,dmModeReadOnly):NULL; 
 
  if(pDoc->db==0) 
  { 
    MemPtrFree(pDoc); 
    *pO=NULL; 
    return ENOENT; 
  } 
 
  // read Header
  pDoc->hFile = DmQueryRecord(pDoc->db,0); 
  pDoc->pHeader = MemHandleLock(pDoc->hFile); 
 
  if(pDoc->pHeader->compressed==0x0002) 
  {
    pDoc->buffer=MemPtrNew(UncBufSize);
    if(pDoc->buffer==NULL)
    {
      MemPtrFree(pDoc); 
      *pO=NULL; 
      return ENOMEM;
    }
    pDoc->mode=COMPRESSED;
  } 
  else pDoc->mode=0;

  // discard header (no longer needed) 
  MemHandleUnlock(pDoc->hFile); 
 
  pDoc->nRecs = DmNumRecords(pDoc->db); pDoc->iRec=1; 
  pDoc->hFile=NULL;
  pDoc->iRec=0;
  pDoc->RecSize=0; // force load record 
  return 0; 
 
} 
 
Err newdoc(UInt16 refnum,DOCFILE**pO,CharPtr fname) 
{ 
 
  int err; 
  LocalID id; 
  VoidPtr p; 
  int n=-1; 
  DOCFILE *pDoc; 
 
  pDoc = MemPtrNew(sizeof(DOCFILE)); 
  *pO=pDoc; 
  if (pDoc==NULL) return ENOMEM; 
 
  pDoc->pHeader = MemPtrNew(sizeof(DOCHEADER)); 
  if (pDoc->pHeader==NULL) 
  { 
    MemPtrFree(pDoc); 
    *pO=NULL; 
    return ENOMEM; 
  } 
 
  pDoc->hBuff = MemHandleNew(DocBufSize); 
  if(pDoc->hBuff==NULL) 
  { 
    MemPtrFree(pDoc->pHeader); 
    MemPtrFree(pDoc); 
    *pO=NULL; 
    return ENOMEM; 
  } 
  pDoc->buffer = MemHandleLock(pDoc->hBuff); 
 
  id = DmFindDatabase(0,fname); 
  if(id) DmDeleteDatabase(0,id); 
 
  err= DmCreateDatabase(0,fname,'REAd','TEXt',false); 
  id = err?0:DmFindDatabase(0,fname); 
  pDoc->db = id?DmOpenDatabase(0,id,dmModeReadWrite):NULL; 
 
  if (pDoc->db==NULL) 
  { 
    MemHandleUnlock(pDoc->hBuff); 
    MemHandleFree(pDoc->hBuff); 
    MemPtrFree(pDoc->pHeader); 
    MemPtrFree(pDoc); 
    *pO=NULL; 
    return EIO; 
  } 
 
  pDoc->nRecs = -1; // new file 
  pDoc->iBuff = 0;  // 1st byte 
  *pDoc->buffer = 0; 
  pDoc->mode=0;
 
  // initialise header 
  pDoc->pHeader->compressed = 0x0001; 
  pDoc->pHeader->spare = 0x0000; 
  pDoc->pHeader->DocSize = 0x00000000; 
  pDoc->pHeader->rNum = 0x0000; 
  pDoc->pHeader->rSize = 0x1000; 
  pDoc->pHeader->position = 0x00000000; 
 
  return 0; 
 
} 
 
// write characters on a new doc mode open doc 
Err writedoc(UInt16 refnum, DOCFILE *pDoc, CharPtr t) 
{ 
 
  VoidPtr p; 
  int n; 
 
  while(*t)  
  { 
 
    pDoc->buffer[pDoc->iBuff++] = *t++; 
 
    // flush + new record 
    if(pDoc->iBuff>=DocBufSize)  
    {
      n=-1;
      
      pDoc->hFile = DmNewRecord(pDoc->db,&n,pDoc->iBuff);

      if(pDoc->hFile==NULL) return ENOSPC;
 
      p = MemHandleLock(pDoc->hFile);
      DmWrite(p,0,pDoc->buffer,pDoc->iBuff); 
      MemHandleUnlock(pDoc->hFile);
      DmReleaseRecord(pDoc->db,n,true);
 
      pDoc->pHeader->DocSize += pDoc->iBuff; 
      pDoc->iBuff=0; 
      *pDoc->buffer=0; 
 
    } 
  } 
  return 0; 
}

Err FPutS(UInt16 refnum,CharPtr s,DOCFILE*pf)
{
 switch(pf->nRecs)
 {
  case -2:
   return printstdout(refnum,s);
  case -1:
   return writedoc(refnum,pf,s);
  default:
   return EIO;
 }
}

// read a line in a read mode open doc 
Err readdocline(UInt16 refnum, DOCFILE *pDoc, int len, char*buffer) 
{ 
  UInt i=0; 
  int c=0; 
  buffer[i]=0;
  int err; 
 
  if(pDoc->nRecs==0) return 0; 
 
  while(c!=0x0a&&i<len) 
  { 

    err=GetDocChar(pDoc,&c);
    if(err)
    {
     *buffer=0;
     return err;
    }
    if(c==EOF) break;
    buffer[i++]=c;
 
  }  
 
  buffer[i]=0; 
  return 0; 
 
} 

Err freadoc(UInt16 refnum,void *pData,unsigned int size,unsigned int blocks,DOCFILE *pF,int *r) 
{

  int c,err;
  unsigned char *pD;
  unsigned int s,f;
  UInt32 p=0;
  
  pD=pData;

  for(f=0;f<blocks;f++) {
    for(s=0;s<size;s++) {
      err=GetDocChar(pF,&c);
      if(err) {
        *r=f;
        return err;
      }
      pD[p++]=(unsigned char)c;
    }
  }
  *r=f;
  return 0;
}   

Err fwritedoc(UInt16 refnum,void *pData,unsigned int size,unsigned int blocks,DOCFILE *pF,int *r)
{

  int err;
  unsigned char *pD, c;
  unsigned int f,s;

  pD=pData;

  for(f=0;f<blocks;f++) {
    for(s=0;s<size;s++) {
      c=pD[s];
      err=WriteDocChar(refnum,pF,c);
      if(err) {
        *r=f;
        return err;
      }
    }
  }
  *r=f;
  return 0;

}

//--------------------------------------
//               STDIN
//--------------------------------------


Err openstdin(UInt16 refnum,DOCFILE**ppF)
{
  DOCFILE*pf;
  pf=MemPtrNew(sizeof(DOCFILE));
  if(pf==NULL)
  {
   *ppF=NULL;
   return ENOMEM;
  }
  pf->nRecs=-3; // stdin mode
  pf->iBuff=0;
  pf->hBuff=MemHandleNew(80);
  if(pf->hBuff==NULL)
  {
   MemPtrFree(pf);
   *ppF=NULL;
   return ENOSPC;
  }
  pf->buffer=MemHandleLock(pf->hBuff);
  *ppF=pf;
  return 0;
}

Err closestdin(UInt16 refnum,DOCFILE*pf)
{
  if(pf)
  {
   if(pf->hBuff)
   {
    MemHandleUnlock(pf->hBuff);
    MemHandleFree(pf->hBuff);
   }
   MemPtrFree(pf);
  }
  return 0;
}

Err getcharin(UInt16 refnum,DOCFILE*pf,int*c)
{
 int ch;
 int err=0;

 if(pf->nRecs==-3)
 { // stdin
  while(pf->iBuff==0)
  {
   err=getstdin(refnum,pf->buffer);
   pf->iBuff=StrLen(pf->buffer);
   pf->pData=pf->buffer;
  }
  ch=*pf->pData++;
  pf->iBuff--;
  // Control Z on stdin is EOF
  if(ch==26) *c=EOF; else *c=ch;
 }
 else if(pf->nRecs==0) *c=EOF;
 else if(pf->nRecs>0)
 { // doc
   err=GetDocChar(pf,c);
 }
 else *c=EOF;
 return err;
}

Err FGetS(UInt16 refnum,char*s,int n,DOCFILE*pf)
{
 int err;
 switch(pf->nRecs)
 {
  case -3:
   return getstdin(refnum,s);
  case -2:
  case -1:
   *s=0;
   return EIO;
  default:
   err=readdocline(refnum,pf,n,s);
   if(err) return err;
   if(*s==0) return EOF;
   return 0;
 }
}

//--------------------------------------
//             Database
//--------------------------------------

 
Err deletedb(UInt16 refnum,char *name) 
{ 
 
  LocalID id; 
 
  id=DmFindDatabase(0,name); 
  if(id) DmDeleteDatabase(0,id); 
  return 0;
} 
 
Err newdb(UInt16 refnum, DmOpenRef**pdb,char *fname,ULong creator,ULong type) 
{ 
 
  int err; 
  LocalID id; 
  UInt attr; 
 
  deletedb(refnum,fname); 
 
  err= DmCreateDatabase(0,fname,creator,type,false); 
 
  if(err==0) { 
    id = DmFindDatabase(0,fname); 
 
    DmDatabaseInfo(0,id,NULL,&attr,NULL,NULL, 
      NULL,NULL,NULL,NULL,NULL,NULL,NULL); 
 
    attr=attr|0x008; // set backup bit 
    DmSetDatabaseInfo(0,id,NULL,&attr,NULL,NULL, 
      NULL,NULL,NULL,NULL,NULL,NULL,NULL); 
     
    *pdb = DmOpenDatabase(0,id,dmModeReadWrite); 
    return 0; 
  } 
  *pdb=NULL;
  return err; 
 
} 

Err opendb(UInt16 refnum,DmOpenRef**pdb,char *fname) 
{ 
 
  LocalID id; 
 
  id=DmFindDatabase(0,fname); 
  if(id) 
  {
    *pdb=DmOpenDatabase(0,id,dmModeReadWrite); 
    return 0;
  }
  *pdb=NULL;
  return 1; 
 
} 
 
// fixed point representation
// integer part must fit in a 32 bit integer
Err FixToStr(UInt refnum,double x,char*s,Int32 d)
{
 UInt32 i;
 int k;

 *s=0;
 if ((d<0)||(d>15)) return 1;
 if(x>=0)
 {
   if(x>0x7fffffff) return 1;
   i=x;
   x=x-i;
 }
 else
 {
  x=-x;
  if(x>0x7fffffff) return 1;
  i=x;
  x=x-i;
  i=-i;
 }
 longtoa(refnum,i,s,10);
 s+=StrLen(s);
 *s++='.';
 for(k=d-1;k>0;k--)
 {
  x=x*10;
  i=x;
  *s++='0'+i;
  x=x-i;
 }
 i=x*10+0.5; // round last digit
 *s++='0'+i;
 *s=0;
 return 0;
}

// scan string s and return integral value
// e is set to the first non used char
// if base <= 0 interpret first char
// 0x... hexa, 0... octal, 1..0 decimal
Err StrToL(UInt refnum,Int32*y,char*s,char**e,int base)
{
 UInt32 x;
 char*t;
 int c;
 Boolean neg;

 x=0;
 t=s;

 c=*t;
 neg=false;
 if(c=='-')
 {
  neg=true;
  c=*++t;
 }
 else if(c=='+') c=*++t;
 if (base<=0)
 {
  if(c=='0')
  {
   c=*++t;
   if(c=='x'||c=='X')
   {
    c=*++t;
    c=radix(c);
    if(c<16)
    {
     base=16;
     goto convert;
    }
    else *e=t-1;
   }
   else
   {
    c=radix(c);
    if(c<8)
    {
     base=8;
     goto convert;
    }
    else *e=t;
   }
  }
  else
  {
   c=radix(c);
   if(c<10)
   {
    base=10;
    goto convert;
   }
   else *e=s;
  }
 }
 else if(base<=36)
 {
  c=radix(c);
  if(c<base)
  {
  convert:
   x=c;
   c=radix(*++t);
   while(c<base)
   {
    x=x*base+c;
    c=radix(*++t);
   }
   *e=t;
  }
  else
  {
    *e=s;
    *y=0;
    return 1;
  }
 }
 if(neg) *y=-x; else *y=x;
 return 0;
}

// scan seauence of decimal digits for integer part
double integer(char*s,char**e)
{
 int j;
 double d;

 d=0.0;
 while(1)
 {
  j=radix(*s);
  if(j>=10) break;
  d=d*10 + j;
  s++;
 }
 *e=s;
 return d;
}

// scan seauence of 0..9 digits for decimal part
double decimal(char*s,char**e)
{
 int j;
 double r;
 double d;
 r=0.0;
 d=1.0;
 while(1)
 {
  j=radix(*s);
  if(j>=10) break;
  s++;
  d=d/10;
  r=r+j*d;
 }
 *e=s;
 return r;
}

Err StrToD(UInt refnum,double*y,char*s,char**e)
{
 double i; // integer
 double d; // decimal
 Int32  x; // exponemt
 Boolean neg;
 int c;
 char*t;
 Err err;

 t=s;
 c=*t;
 d=0;
 x=0;
 neg=false;
 if(c=='-')
 {
  c=*++t;
  neg=true;
 }
 else if(c=='+') c=*++t;
 if(c=='.')
 {
  c=*++t;
  if(radix(c)<10) { i=0; goto dec; }
 error:
  *e=s;
  *y=0;
  return 1;
 }
 if(radix(c)>=10) goto error;
 i=integer(t,e);
 t=*e;
 c=*t;
 switch(c)
 {
  case 'e':
  case 'E':
   d=0;
   goto exp;
  case '.':
   c=*++t;
   if(radix(c)<10)
   {
   dec:
    d=decimal(t,e);
    t=*e;
    c=*t;
   }
   if ((c=='e')||(c=='E'))
   {
   exp:
    StrToL(refnum,&x,t+1,e,10);
   }
   else *e=t;
  default:
  cal:
   if(x<0) *y=(i+d)/Pow10(-x);
   else *y=(i+d)*Pow10(x);
   if(neg) *y=-*y;
 }
 return 0;
}

